(function () {
    'use strict';

    const ICONS = {
        code: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="16 18 22 12 16 6"></polyline><polyline points="8 6 2 12 8 18"></polyline></svg>`,
        github: `<svg viewBox="0 0 24 24" fill="currentColor"><path d="M12 0c-6.626 0-12 5.373-12 12 0 5.302 3.438 9.8 8.207 11.387.599.111.793-.261.793-.577v-2.234c-3.338.726-4.033-1.416-4.033-1.416-.546-1.387-1.333-1.756-1.333-1.756-1.089-.745.083-.729.083-.729 1.205.084 1.839 1.237 1.839 1.237 1.07 1.834 2.807 1.304 3.492.997.107-.775.418-1.305.762-1.604-2.665-.305-5.467-1.334-5.467-5.931 0-1.311.469-2.381 1.236-3.221-.124-.303-.535-1.524.117-3.176 0 0 1.008-.322 3.301 1.23.957-.266 1.983-.399 3.003-.404 1.02.005 2.047.138 3.006.404 2.291-1.552 3.297-1.23 3.297-1.23.653 1.653.242 2.874.118 3.176.77.84 1.235 1.911 1.235 3.221 0 4.609-2.807 5.624-5.479 5.921.43.372.823 1.102.823 2.222v3.293c0 .319.192.694.801.576 4.765-1.589 8.199-6.086 8.199-11.386 0-6.627-5.373-12-12-12z"/></svg>`,
        paste: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M16 4h2a2 2 0 0 1 2 2v14a2 2 0 0 1-2 2H6a2 2 0 0 1-2-2V6a2 2 0 0 1 2-2h2"></path><rect x="8" y="2" width="8" height="4" rx="1" ry="1"></rect></svg>`,
        upload: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path><polyline points="17 8 12 3 7 8"></polyline><line x1="12" y1="3" x2="12" y2="15"></line></svg>`,
        file: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M13 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V9z"></path><polyline points="13 2 13 9 20 9"></polyline></svg>`
    };

    window.addEventListener("PCH_CONFIG", (e) => {
        if (e.detail.pch_editor_enabled) {
            new PCHEditor(e.detail);
        }
    });

    window.addEventListener("PCH_PASTE_DATA", (e) => {
        if (e.detail) {
            new PCHEditor({}).submitCode(e.detail);
        }
    });

    class PCHEditor {
        constructor(settings) {
            this.settings = settings;
            this.aceEditor = null;
            this.retryCount = 0;
            this.langSelect = null;
            this.dragOverlay = null;
            this.tryInit();
            this.setupGlobalPasteListener();
            this.setupGlobalDragListener();
        }

        tryInit() {
            const target = document.querySelector('.submit__des') || document.querySelector('.problem-content');
            if (document.querySelector('.pch-helper-box')) return;

            if (!window.ace || !target) {
                if (this.retryCount < 20) {
                    this.retryCount++;
                    setTimeout(() => this.tryInit(), 300);
                }
                return;
            }

            const box = this.createEditorBox();

            let insertAfter = null;
            const parent = target.parentNode;
            if (parent) {
                const walker = document.createTreeWalker(parent, NodeFilter.SHOW_TEXT);
                let node;
                while (node = walker.nextNode()) {
                    if (node.textContent.includes('Giới hạn')) {
                        insertAfter = node.parentElement;
                    }
                }
            }

            if (insertAfter?.nextSibling) {
                insertAfter.parentNode.insertBefore(box, insertAfter.nextSibling);
            } else {
                target.parentNode.insertBefore(box, target.nextSibling);
            }

            this.initAce();
            setTimeout(() => this.createDragOverlay(), 100);
        }

        setupGlobalPasteListener() {
            document.addEventListener('paste', (e) => {
                if (!this.aceEditor) return;
                const aceTextInput = document.querySelector('.ace_text-input');
                if (aceTextInput && document.activeElement === aceTextInput) return;

                const items = e.clipboardData?.items;
                if (!items) return;

                for (let item of items) {
                    if (item.kind === 'file') {
                        e.preventDefault();
                        const file = item.getAsFile();
                        if (file) this.handleFileSelect(file);
                        break;
                    }
                }
            });
        }

        setupGlobalDragListener() {
            let dragCounter = 0;

            document.addEventListener('dragenter', (e) => {
                if (e.dataTransfer?.types.includes('Files')) {
                    dragCounter++;
                    if (this.dragOverlay) {
                        this.dragOverlay.style.display = 'flex';
                    }
                }
            });

            document.addEventListener('dragleave', (e) => {
                dragCounter--;
                if (dragCounter === 0 && this.dragOverlay) {
                    this.dragOverlay.style.display = 'none';
                }
            });

            document.addEventListener('drop', (e) => {
                dragCounter = 0;
                if (this.dragOverlay) {
                    this.dragOverlay.style.display = 'none';
                }
            });
        }

        createDragOverlay() {
            this.dragOverlay = document.createElement('div');
            this.dragOverlay.className = 'pch-drag-overlay';
            this.dragOverlay.innerHTML = `
                <div class="pch-drag-content">
                    ${ICONS.upload}
                    <div class="pch-drag-text">Thả để nhập vào Code Editor</div>
                </div>
            `;

            this.dragOverlay.addEventListener('dragover', (e) => {
                e.preventDefault();
                e.stopPropagation();
            });

            this.dragOverlay.addEventListener('drop', (e) => {
                e.preventDefault();
                e.stopPropagation();
                this.dragOverlay.style.display = 'none';

                const files = e.dataTransfer?.files;
                if (files && files.length > 0) {
                    this.handleFileSelect(files[0]);
                }
            });

            document.body.appendChild(this.dragOverlay);
        }

        handleFileSelect(file) {
            if (!file || !this.aceEditor) return;
            const reader = new FileReader();
            reader.onload = (e) => this.aceEditor.setValue(e.target.result, -1);
            reader.onerror = () => alert('Không thể đọc tệp!');
            reader.readAsText(file);
        }

        createEditorBox() {
            const container = document.createElement('div');
            container.className = 'pch-helper-box';

            const header = document.createElement('div');
            header.className = 'pch-box-header';
            header.innerHTML = `<div class="pch-box-title">${ICONS.code} PTIT Code Helper</div>`;

            const toolbar = document.createElement('div');
            toolbar.className = 'pch-toolbar';

            this.langSelect = document.createElement('select');
            this.langSelect.className = 'pch-btn';
            const siteCompiler = document.querySelector('#compiler');
            if (siteCompiler) {
                Array.from(siteCompiler.options).forEach(opt => {
                    const myOpt = document.createElement('option');
                    myOpt.value = opt.value;
                    myOpt.text = opt.text;
                    this.langSelect.add(myOpt);
                });
                this.langSelect.value = siteCompiler.value;
                this.langSelect.onchange = () => {
                    siteCompiler.value = this.langSelect.value;
                    siteCompiler.dispatchEvent(new Event('change'));
                    this.updateAceMode();
                };
            }
            toolbar.appendChild(this.langSelect);

            const quickBtn = document.createElement('button');
            quickBtn.className = 'pch-btn';
            quickBtn.innerHTML = `${ICONS.paste} Nộp code từ bộ nhớ tạm`;
            quickBtn.onclick = () => {
                navigator.clipboard.readText()
                    .then(text => {
                        this.submitCode(text);
                    })
                    .catch(() => {
                        window.dispatchEvent(new CustomEvent("PCH_REQ_CLIPBOARD"));
                    });
            };
            toolbar.appendChild(quickBtn);

            const submitBtn = document.createElement('button');
            submitBtn.className = 'pch-btn pch-btn-red';
            submitBtn.textContent = 'Nộp code trong editor';
            submitBtn.onclick = () => this.submitCode(this.aceEditor.getValue());
            toolbar.appendChild(submitBtn);

            header.appendChild(toolbar);
            container.appendChild(header);

            const editorDiv = document.createElement('div');
            editorDiv.id = 'pch-ace-editor';
            container.appendChild(editorDiv);

            return container;
        }

        initAce() {
            try {
                this.aceEditor = ace.edit("pch-ace-editor");
                this.aceEditor.setTheme("ace/theme/textmate");
                this.aceEditor.setFontSize(14);
                this.aceEditor.setShowPrintMargin(false);
                if (this.langSelect) this.updateAceMode();
            } catch (e) { console.error("Ace Editor init failed", e); }
        }

        updateAceMode() {
            if (!this.aceEditor || !this.langSelect) return;
            const text = this.langSelect.options[this.langSelect.selectedIndex].text;
            let mode = "c_cpp";
            if (text.includes("Java")) mode = "java";
            else if (text.includes("Python")) mode = "python";
            else if (text.includes("Golang")) mode = "golang";
            else if (text.includes("C#")) mode = "csharp";
            this.aceEditor.session.setMode("ace/mode/" + mode);
        }

        submitCode(code) {
            if (!code || !code.trim()) return alert("Code trống!");

            const fileInput = document.querySelector("input[type='file']");
            const submitBtn = document.querySelector(".submit__pad__btn") || document.querySelector(".submit-btn-real");

            if (!fileInput || !submitBtn) return alert("Không tìm thấy form nộp bài gốc!");

            const file = new File([code], "solution.cpp", { type: "text/plain" });
            const dt = new DataTransfer();
            dt.items.add(file);
            fileInput.files = dt.files;
            fileInput.dispatchEvent(new Event("change", { bubbles: true }));

            const oldText = submitBtn.innerText;
            submitBtn.innerText = "Đang nộp...";
            setTimeout(() => {
                submitBtn.click();
                submitBtn.innerText = oldText;
            }, 500);
        }
    }
})();
